# R functions to run R - Needs to be loaded before running Eva from R
.GlobalEnv
##########################################################################
EvaSetDefaultParameters <- function() {
  DataParameters <- list( DataFile = "eva.dat", ResultsDirectory = "EVA_", 
                          IgnoreParentalPedigreeErrors = ".false.", RecodeFile = "_null_")
  
  PopulationHistory <- list( PCI_ngen = 5)
  
  Contribution <- list( ancestor = 0, descendant = 0, group = 0)
  
  RelationshipMatrix <- list( source = "pedigree", Gfile = "_null_", Timesteps = 0)
  
  OCSParameters <- list( Nmatings = -1, optimise ="penalty", w_merit = 0, w_relationship = -1,
                         dFConstraint = 1.0 ) #, W_nmales = 0.0, NSelectedMales = 0)
  
  AlgorithmParameters <- list( generations = 10000, popsize = 100, n_offspring = 10,
                               restart_interval =1000, exchange_algorithm = 1500,
                               mutate_probability =1.0, crossover_probability = 1.0, 
                               directed_mutation_probability = -1.0, 
                               seed_rng =0, NgenerationsNoImprovement = 10000)
  
  MatingOptions <- list( MatingsStrategy = "random", RepeatedMatings = ".true.")    
  
  #EvaParm <<- list(DataParameters=DataParameters, PopulationHistory=PopulationHistory, 
  #                Contribution=Contribution, RelationsMatrix=RelationshipMatrix,
  #                OCSParameters=OCSParameters, AlgorithmParameters=AlgorithmParameters, 
  #                MatingOptions=MatingOptions)  
  assign("EvaParm", list(DataParameters=DataParameters, PopulationHistory=PopulationHistory, 
              Contribution=Contribution, RelationsMatrix=RelationshipMatrix,
              OCSParameters=OCSParameters, AlgorithmParameters=AlgorithmParameters, 
              MatingOptions=MatingOptions), envir = .GlobalEnv)
  
  remove(DataParameters,PopulationHistory,Contribution,RelationshipMatrix, 
         OCSParameters,AlgorithmParameters,MatingOptions)
}

###################################################################
EvaSaveParameterFile <- function(fname) {
  save(EvaParm, file = "EvaParm.Rda")
}

###################################################################
EvaLoadParameterFile <- function(fname) {
  load(fname, envir = .GlobalEnv )
}

###################################################################
EvaRun <- function() {
 
  prmfil <- file("eva.prm","w")
  cat("&DataParameters \n", file=prmfil)
  cat(paste("  DataFile = ",gsub(" ","",paste("'",EvaParm$DataParameters$DataFile[1], "'")) ,", \n"), file=prmfil)
  cat(paste("  ResultsDirectory= ",gsub(" ","",paste("'",EvaParm$DataParameters$ResultsDirectory[1],"'")), ", \n"),file=prmfil)
  cat(paste("  IgnoreParentalPedigreeErrors=", gsub(" ","", paste(EvaParm$DataParameters$IgnoreParentalPedigreeErrors)),", \n"), file=prmfil)
  cat(paste("  RecodeFile=", gsub(" ", "", paste("'",EvaParm$DataParameters$RecodeFile,"'"))," /  \n"), file=prmfil)
    
  cat("&PopulationHistory  \n", file=prmfil)
  cat("  PCI_ngen = ",EvaParm$PopulationHistory$PCI_ngen,"  / \n", file=prmfil)
  
  for (i in 1:length(EvaParm$Contribution$ancestor)) {
    cat("&Contribution   ancestor=",EvaParm$Contribution$ancestor[i],
        " , descendant =",EvaParm$Contribution$descendant[i]," , group=",EvaParm$Contribution$group[i], " /  \n", file=prmfil)
    
  }
  
  cat("&RelationshipMatrix \n", file=prmfil)
  cat("  source = ",gsub(" ","",paste("'",EvaParm$RelationsMatrix$source, "'"))," , \n ", file=prmfil)
  cat(paste("  GFile = ",gsub(" ","",paste("'",EvaParm$RelationsMatrix$Gfile, "'")), " , \n "), file=prmfil)
  cat("  Timesteps = ", EvaParm$RelationsMatrix$Timesteps ," /  \n", file=prmfil)
  
  cat("&OCSParameters \n", file=prmfil)
  cat("  nmatings         = ", EvaParm$OCSParameters$Nmatings ,", \n", file=prmfil)
  cat(paste("  optimise         = ", gsub(" ","",paste("'",EvaParm$OCSParameters$optimise,"'"))," , \n"), file=prmfil)
  cat("  w_merit          = ", EvaParm$OCSParameters$w_merit ,", \n", file=prmfil)
  cat("  w_relationship   = ", EvaParm$OCSParameters$w_relationship, " ,  \n", file=prmfil)
  cat("  dFConstraint     = ", EvaParm$OCSParameters$dFConstraint, " , / \n", file=prmfil)
  #cat("  W_nmales         = ", EvaParm$OCSParameters$W_nmales, " , \n", file=prmfil)
  #cat("  NSelectedMales   = ", EvaParm$OCSParameters$NSelectedMales, " / \n", file=prmfil)
  
  cat("&AlgorithmParameters  \n", file=prmfil)
  cat("  generations  = ",EvaParm$AlgorithmParameters$generations," ,   \n", file=prmfil)
  cat("  popsize      = ",EvaParm$AlgorithmParameters$popsize," ,   \n", file=prmfil)
  cat("  n_offspring  = ",EvaParm$AlgorithmParameters$n_offspring," ,   \n", file=prmfil)
  cat("  restart_interval      = ",EvaParm$AlgorithmParameters$restart_interval," ,   \n", file=prmfil)
  cat("  exchange_algorithm    = ",EvaParm$AlgorithmParameters$exchange_algorithm," ,   \n", file=prmfil)
  cat("  mutate_probability    = ",EvaParm$AlgorithmParameters$mutate_probability," ,   \n", file=prmfil)
  cat("  crossover_probability = ",EvaParm$AlgorithmParameters$crossover_probability," ,   \n", file=prmfil)
  cat("  directed_mutation_probability  = ",EvaParm$AlgorithmParameters$directed_mutation_probability," ,   \n", file=prmfil)
  cat("  seed_rng  = ",EvaParm$AlgorithmParameters$seed_rng," ,   \n", file=prmfil)
  cat("  NgenerationsNoImprovement       = ",EvaParm$AlgorithmParameters$NgenerationsNoImprovement," /   \n", file=prmfil)
  
  cat("&MatingOptions \n", file=prmfil)
  cat(paste("  MatingStrategy = ",gsub(" ","",paste("'",EvaParm$MatingOptions$MatingsStrategy, "'")), " , \n"), file=prmfil)
  cat(paste("  RepeatedMatings = ",EvaParm$MatingOptions$RepeatedMatings, " / \n"), file=prmfil)
  
  close(prmfil) # tidy up

  # run eva
  if (.Platform$OS.type == "windows") {
    # windows machine
    pth <- Sys.getenv("EVAhome")
    # pth <- Sys.getent("ProgramFiles(x86)")
    pth <- paste(pth,"eva.exe", sep = "")
    system2(command = pth, args = "eva.prm", stdout = TRUE, wait = TRUE)
  } else {
    # other (Mac or Linux)
    system2(command="eva", args="eva.prm", wait=TRUE)
  }
}

###################################################################
EvaLoadResultFiles <- function() {
  # Load file with inbreeding coefficientas
  mfile <- gsub(" ","",paste(EvaParm$DataParameters$ResultsDirectory,"f_coeff.txt"))
  if (file.exists(mfile)) {
    assign("fFcoeff", read.table(file=mfile, header=TRUE, skip=3), envir = .GlobalEnv)
  }

  # Load file with genetic contributions
  mfile <- gsub(" ","",paste(EvaParm$DataParameters$ResultsDirectory,"gencont.txt"))
  if (file.exists(mfile)) {
    assign("fGencont", read.table(file=mfile, header=TRUE, skip=3, 
           col.names = c("Ancestor","Descendant","Group","N.Group","Add.Rel",
                         "Gen.Cont","text","Ancestor")),
           envir = .GlobalEnv)
  }
  
  # Load file with largest genetic contributions
  mfile <- gsub(" ","",paste(EvaParm$DataParameters$ResultsDirectory,"max_gc.txt"))
  if (file.exists(mfile)) {
    assign("fMaxGC", read.table(file=mfile, header=TRUE, skip=4, 
                                col.names = c("Ancestor","Group","N.Group","Add.Rel",
                                              "Gen.Cont","text")) , 
            envir = .GlobalEnv)
      }
  
  # Load file with candidates
  mfile <- gsub(" ","",paste(EvaParm$DataParameters$ResultsDirectory,"Candidates.txt"))
  if (file.exists(mfile)) {
    assign("fCandidates", read.table(file=mfile, header=TRUE, skip=6), envir = .GlobalEnv)
    
  }
  
  # Load file with mating list
  mfile <- gsub(" ","",paste(EvaParm$DataParameters$ResultsDirectory,"eva_MatingList.txt"))
  if (file.exists(mfile)) {
    assign("fMatingList", read.table(file=mfile, header=TRUE, skip=0), envir = .GlobalEnv)
  }

  # Load file with male matings
  mfile <- gsub(" ","",paste(EvaParm$DataParameters$ResultsDirectory,"male_list.txt"))
  if (file.exists(mfile)) {
    assign("fMaleList", read.table(file=mfile, header=TRUE, skip=5), envir = .GlobalEnv)
  }
  
}


##################################################################
EvaPlotResults <- function(env=.GlobalEnv) {
 
  # Load file with inbreeding coefficientas
  if (exists("fFcoeff")) {
    library(ggplot2)
    library(hrbrthemes)
    library(tidyverse)
    library(viridis)
    
    pl1 <- ggplot(fFcoeff, aes(x = as.factor(fFcoeff$Group), y = fFcoeff$F)) +
      geom_violin() + 
      ggtitle("Violin plot of individual inbreeding coeffcicients by year")
    assign("plot1",pl1,envir = .GlobalEnv)
    pl2 <- ggplot(fFcoeff, aes(x = as.factor(fFcoeff$Group), y = fFcoeff$PCI5)) +
      geom_violin() + 
      ggtitle("Violin plot of indiidual pedigree completeness by year")
    print("Plotted f violin plot")
    assign("plot2",pl2,envir = .GlobalEnv)
    
    pl3 <- ggplot(fFcoeff,  aes( x= as.factor(Group), y= F)) +
      geom_boxplot(fill="#69b3a2") +
#      scale_fill_viridis(discrete = TRUE, alpha = 0.6) +
#      geom_jitter(color="black", size=0.4, alpha=0.9) +
#      theme_ipsum() +
#      theme( legend.position="none", plot.title= element_text(size=1)) +
      ggtitle("Boxplot of inbreeding coefficients by group") +
      xlab("Group")
    print("Plotted f box-plot by group")
    assign("plot3",pl3,envir = .GlobalEnv)
    
    pl4 <-  ggplot(fFcoeff, aes( x= as.factor(Group), y= PCI5)) +
                  geom_boxplot(fill="#69b3a2") +
      #      scale_fill_viridis(discrete = TRUE, alpha = 0.6) +
      #      geom_jitter(color="black", size=0.4, alpha=0.9) +
#      theme_ipsum() +
      #      theme( legend.position="none", plot.title= element_text(size=1)) +
      ggtitle("Boxplot of pedigree completeness by group") +
      xlab("Group")
    print("Plotted PCI boxplot")
    assign("plot4",pl4,envir = .GlobalEnv)

  }

  # Load file with candidates
  if (exists("fCandidates")) {
    # Most basic bubble plot
    subdat <- fCandidates[fCandidates$N.matings>0,]
    pl5 <- ggplot(subdat, aes(x=Merit, y=Rel.females, size = as.factor(N.matings), color=as.factor(Sex))) +
      geom_point(alpha=0.7) +
      ggtitle("Contributions as a function of merit and relationship to males")
    assign("plot5", pl5, envir = .GlobalEnv)
    pl6 <- ggplot(subdat, aes(x=Merit, y=Rel.females, size = as.factor(N.matings), color=as.factor(Sex))) +
      geom_point(alpha=0.7) +
      ggtitle("Contributions as a function of merit and relationship to females")
    assign("plot6", pl6, envir = .GlobalEnv)
    pl7 <- ggplot(fCandidates, aes(x=Merit, y=Rel.females, size = as.factor(N.matings), color=as.factor(Sex))) +
      geom_point(alpha=0.7) +
      ggtitle("Contributions as a function of merit and relationship to males")
    assign("plot7", pl7, envir = .GlobalEnv)
    pl8 <- ggplot(fCandidates, aes(x=Merit, y=Rel.females, size = as.factor(N.matings), color=as.factor(Sex))) +
      geom_point(alpha=0.7) +
      ggtitle("Contributions as a function of merit and relationship to females")
    assign("plot8", pl8, envir = .GlobalEnv)
    
  }
  
  
}

##################################################################
EvaLog <- function() {
  file.show("eva.log",title = "eva.log")
}

##################################################################
EvaSummary <- function() {
  mfile <- gsub(" ","",paste(EvaParm$DataParameters$ResultsDirectory,"F_summary.txt"))
  file.show(mfile,title = mfile)
}

##################################################################
EvaPlotConvergence <- function() {
  # Load file with best solution
  mfile <- "eva_best.txt"
  if (file.exists(mfile)) {
    fconv <- read.table(file=mfile, header=TRUE, skip=0)
    fconv$Log10Generation <- log10(fconv$Generation)
    fconv$Evaluation <- fconv$Evaluation - min(fconv$Evaluation)
    fconv$Evaluation <- fconv$Evaluation  / max(fconv$Evaluation)
    
    library(ggplot2)
    library(dplyr)
    
    fconv %>%
      #tail(10) %>%
      ggplot( aes(x=Log10Generation, y=Evaluation)) +
      geom_line() +
      geom_point() +
      ggtitle("Convergence: Evaluation vs. log10 Generation")
  }
  
}
